/***************************************************************************
                          drawGrid3D.h  -  description
                             -------------------
    begin                : 2009/07/26
    copyright            : (C) 2009 by Tomas Oberhuber
    email                : tomas.oberhuber@fjfi.cvut.cz
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef drawGrid3DH
#define drawGrid3DH

#include <ostream>
#include <fstream>
#include <iomanip>
#include <float.h>
#include <diff/tnlGrid3D.h>

template< class T > bool Draw( const tnlGrid3D< T >& u,
                               ostream& str,
                               const char* format,
                               const int i_step = 1,
                               const int j_step = 1,
                               const int k_step = 1 )
{
   if( ! format )
   {
      cerr << "No format given for drawing 3D grid. " << endl;
      return false;
   }


   int i, j, k;
   const int x_size = u. GetXSize();
   const int y_size = u. GetYSize();
   const int z_size = u. GetZSize();
   const T& ax = u. GetAx();
   const T& ay = u. GetAy();
   const T& az = u. GetAy();
   const T& hx = u. GetHx();
   const T& hy = u. GetHy();
   const T& hz = u. GetHy();
   if( strncmp( format, "bin", 3 ) == 0 )
   {
      if( ! u. Save( str ) ) return false;
      return true;
   }
   if( strcmp( format, "gnuplot" ) == 0 )
   {
      cout << "GNUPLOT is not supported for tnlGrid3D." << endl;
      return false;
   }
   if( strcmp( format, "vti" ) == 0 )
   {
      str << "<VTKFile type=\"ImageData\" version=\"0.1\" byte_order=\"LittleEndian\">" << endl;
      str << "<ImageData WholeExtent=\"" 
           << "0 " << x_size - 1 
           << " 0 " << y_size - 1
           << " 0 " << z_size - 1 << "\" Origin=\"0 0 0\" Spacing=\"" 
           << hx << " " << hy << " " << hz << "\">" << endl;
      str << "<Piece Extent=\"0 "
          << x_size - 1 << " 0 " 
          << y_size - 1 << " 0 "
          << z_size - 1 << "\">" << endl;
      str << "<PointData Scalars=\"order_parameter\">" << endl;
      str << "<DataArray Name=\"order_parameter\" type=\"Float32\" format=\"ascii\">" << endl;
      str. flags( ios_base::scientific );
      int i, j, k;
      for( k = 0; k <= z_size - k_step; k += k_step )
         for( j = 0; j <= y_size - j_step; j += j_step )
            for( i = 0; i <= x_size - i_step; i += i_step )
            {
              str << u( i, j, k ) << " ";
            }
      str << endl;
      str << "</DataArray>" << endl;
      str << "</PointData>" << endl;
      str << "</Piece>" << endl;
      str << "</ImageData>" << endl;
      str << "</VTKFile>" << endl;
      return true;
   }
   if( strcmp( format, "povray" ) == 0 )
   {
      str. put( ( char ) ( x_size >> 8 ) );
      str. put( ( char ) ( x_size & 0xff ) );
      str. put( ( char ) ( y_size >> 8 ) );
      str. put( ( char ) ( y_size & 0xff ) );
      str. put( ( char ) ( z_size >> 8 ) );
      str. put( ( char ) ( z_size & 0xff ) );
      int i, j, k;
      T min( u( 0, 0, 0 ) ), max( u( 0, 0, 0 ) );
      for( k = 0; k < z_size; k ++ ) 
         for( j = 0; j < y_size; j ++ ) 
            for( i = 0; i < x_size; i ++ )
            {
               min = Min( min, u( i, j, k ) );
               max = Max( max, u( i, j, k ) );
            }
            
      for( k = 0; k < z_size; k ++ ) 
         for( j = 0; j < y_size; j ++ ) 
            for( i = 0; i < x_size; i ++ )
            {
               int v = 255.0 * ( u( i, j, k ) - min ) / ( max - min );
               str. write( ( char* ) &v, sizeof( int ) );
            }
      return true;
   }
   cerr << "Unknown format '" << format << "' for drawing a grid 3D." << endl;
   return false;
};

template< class T > bool Draw( const tnlGrid3D< T >& u,
                               const char* file_name,
                               const char* format,
                               const int i_step = 1,
                               const int j_step = 1,
                               const int k_step = 1 )
{
   fstream file;
   if( strncmp( format, "bin",3 ) == 0 )
      file. open( file_name, ios :: out | ios :: binary );
   else file. open( file_name, ios :: out );
   if( ! file )
   {
      cerr << "Sorry I can not open the file " << file_name << endl;
      return false;
   }
   bool result = Draw( u, file, format, i_step, j_step, k_step );
   file. close();
   if( ! result )
      return false;
   int len = strlen( format );
   if( strcmp( format + Max( 0, len - 3 ), "-gz" ) == 0 && 
       ! CompressFile( file_name, "gz" ) )
      return false;
   if( strcmp( format + Max( 0, len - 4 ), "-bz2" ) == 0  &&
       ! CompressFile( file_name, "bz2" ) )
         return false;
   return true;
};

template< class T > bool Read( tnlGrid3D< T >& u,
                               const char* input_file )
{
   int strln = strlen( input_file );
   tnlString uncompressed_file_name( input_file );
   if( strcmp( input_file + strln - 3, ".gz" ) == 0 )
      if( ! UnCompressFile( input_file, "gz" ) )
      {
         cerr << "Unable to uncompress the file " << input_file << "." << endl;
         return false;
      }
      else uncompressed_file_name. SetString( input_file, 0, 3 );
   if( strcmp( input_file + strln - 4, ".bz2" ) == 0 )
      if( ! UnCompressFile( input_file, "bz2" ) )
      {
         cerr << "Unable to uncompress the file " << input_file << "." << endl;
         return false;
      }
      else uncompressed_file_name. SetString( input_file, 0, 4 );

       
   fstream file;
   file. open( uncompressed_file_name. Data(), ios :: in | ios :: binary );
   if( ! file )
   {
      cout << " unable to open file " << uncompressed_file_name. Data() << endl;
      return false;
   }
   if( ! u. Load( file ) )
   {
      cout << " unable to restore the data " << endl;
      return false;
   }
   file. close();
   if( strcmp( input_file + strln - 3, ".gz" ) == 0 &&
       ! CompressFile( uncompressed_file_name. Data(), "gz" ) )
   {
      cerr << "Unable to compress back the file " << input_file << "." << endl;
      return false;
   }
   if( strcmp( input_file + strln - 4, ".bz2" ) == 0 &&
       ! CompressFile( uncompressed_file_name. Data(), "bz2" ) )
   {
      cerr << "Unable to compress back the file " << input_file << "." << endl;
      return false;
   }
   return true;
}


#endif
