/***************************************************************************
                          Array.h -  description
                             -------------------
    begin                : Jul 4, 2012
    copyright            : (C) 2012 by Tomas Oberhuber
    email                : tomas.oberhuber@fjfi.cvut.cz
 ***************************************************************************/

/* See Copyright Notice in tnl/Copyright */

#pragma once

#include <list>
#include <vector>
#include <TNL/Object.h>
#include <TNL/File.h>
#include <TNL/TypeTraits.h>
#include <TNL/Devices/Host.h>

namespace TNL {
/**
 * \brief Namespace for TNL containers.
 */
namespace Containers {

template< int, typename > class StaticArray;

/**
 * \brief Array is responsible for memory management, basic elements 
 * manipulation and I/O operations. 
 *
 * \tparam Value Type of array values.
 * \tparam Device Device type - some of \ref Devices::Host and \ref Devices::Cuda.
 * \tparam Index Type for indexing.
 *
 * In the \e Device type, the Array remembers where the memory is allocated. 
 * This ensures the compile-time checks of correct pointers manipulation.
 * Methods defined as \ref __cuda_callable__ can be called even from kernels
 * running on device. Array elements can be changed either using the \ref operator[]
 * which is more efficient but it can be called from CPU only for arrays
 * allocated on host (CPU) and when the array is allocated on GPU, the operator[]
 * can be called only from kernels running on the device (GPU). On the other
 * hand, methods \ref setElement and \ref getElement, can be called only from the
 * host (CPU) does not matter if the array is allocated on the host or the device.
 * In the latter case, explicit data transfer between host and device (via PCI
 * express or NVlink in more lucky systems) is invoked and so it can be very
 * slow. In not time critical parts of code, this is not an issue, however. 
 * Another way to change data stored in the array is \ref evaluate which evaluates
 * given lambda function. This is performed at the same place where the array is
 * allocated i.e. it is efficient even on GPU. For simple checking of the array
 * contents, one may use methods \ref containValue and \ref containsValue and
 * \ref containsOnlyValue.
 * Array also offers data sharing using methods \ref bind. This is, however, obsolete
 * and will be soon replaced with proxy object \ref ArrayView.
 * 
 * \par Example
 * \include ArrayExample.cpp
 */
template< typename Value,
          typename Device = Devices::Host,
          typename Index = int >
class Array : public Object
{
   public:

      using ValueType = Value;
      using DeviceType = Device;
      using IndexType = Index;
      using ThisType = Containers::Array< ValueType, DeviceType, IndexType >;
      using HostType = Containers::Array< Value, Devices::Host, Index >;
      using CudaType = Containers::Array< Value, Devices::Cuda, Index >;

      /** \brief Basic constructor.
       *
       * Constructs an empty array with the size of zero.
       */
      Array();

      /**
       * \brief Constructor with size.
       *
       * \param size Number of array elements. / Size of allocated memory.
       */
      Array( const IndexType& size );

      /**
       * \brief Constructor with data and size.
       *
       * \param data Pointer to data.
       * \param size Number of array elements.
       */
      Array( Value* data,
             const IndexType& size );

      /**
       * 
       * @param 
       */
      // Deep copy does not work because of EllpackIndexMultiMap - TODO: Fix it
      //Array( const Array& );

      /**
       * \brief Copy constructor.
       *
       * The constructor does not make a deep copy, but binds to the supplied array.
       * \param array Existing array that is to be bound.
       * \param begin The first index which should be bound.
       * \param size Number of array elements that should be bound.
       */
      Array( Array& array,
             const IndexType& begin = 0,
             const IndexType& size = 0 );

      Array( Array&& );

      /**
       * \brief Initialize the array from initializer list, i.e. { ... }
       * 
       * @param list Initializer list.
       */
      template< typename InValue >
      Array( const std::initializer_list< InValue >& list );

      /**
       * \brief Initialize the array from std::list.
       * 
       * @param list Input STL list.
       */
      template< typename InValue >
      Array( const std::list< InValue >& list );

      /**
       * \brief Initialize the array from std::vector.
       * 
       * @param vector Input STL vector.
       */
      template< typename InValue >
      Array( const std::vector< InValue >& vector );

      /**
       *  \brief Returns type of array Value, Device type and the type of Index. 
       */
      static String getType();

      /**
       * \brief Returns type of array Value, Device type and the type of Index. 
       */
      virtual String getTypeVirtual() const;

      /**
       *  \brief Returns (host) type of array Value, Device type and the type of Index. 
       */
      static String getSerializationType();

      /**
       *  \brief Returns (host) type of array Value, Device type and the type of Index.
       */
      virtual String getSerializationTypeVirtual() const;

      /**
       * \brief Method for setting the size of an array.
       *
       * If the array shares data with other arrays these data are released.
       * If the current data are not shared and the current size is the same
       * as the new one, nothing happens.
       *
       * \param size Number of array elements.
       */
      void setSize( Index size );

      /** \brief Method for getting the size of an array. */
      __cuda_callable__ Index getSize() const;

      /**
       * \brief Assigns features of the existing \e array to the given array.
       *
       * Sets the same size as the size of existing \e array.
       * \tparam ArrayT Type of array.
       * \param array Reference to an existing array.
       */
      template< typename ArrayT >
      void setLike( const ArrayT& array );

      /**
       * \brief Binds \e _data with this array.
       *
       * Releases old data and binds this array with new \e _data. Also sets new
       * \e _size of this array.
       * @param _data Pointer to new data.
       * @param _size Size of new _data. Number of elements.
       */
      void bind( Value* _data,
                 const Index _size );

      /**
       * \brief Binds this array with another \e array.
       *
       * Releases old data and binds this array with new \e array starting at
       * position \e begin. Also sets new \e size of this array.
       * \tparam ArrayT Type of array.
       * \param array Reference to a new array.
       * \param begin Starting index position.
       * \param size Size of new array. Number of elements.
       */
      template< typename ArrayT >
      void bind( const ArrayT& array,
                 const IndexType& begin = 0,
                 const IndexType& size = 0 );

      /**
       * \brief Binds this array with a static array of size \e Size.
       *
       * Releases old data and binds this array with a static array of size \e
       * Size.
       * \tparam Size Size of array.
       * \param array Reference to a static array.
       */
      template< int Size >
      void bind( StaticArray< Size, Value >& array );

      /**
       * \brief Swaps all features of given array with existing \e array.
       *
       * Swaps sizes, all values (data), allocated memory and references of given
       * array with existing array.
       * \param array Existing array, which features are swaped with given array.
       */
      void swap( Array& array );

      /**
       * \brief Resets the given array.
       *
       * Releases all data from array.
       */
      void reset();

      /**
       * \brief Method for getting the data from given array with constant poiner.
       */
      __cuda_callable__ const Value* getData() const;

      /**
       * \brief Method for getting the data from given array.
       */
      __cuda_callable__ Value* getData();

      /**
       * \brief Assignes the value \e x to the array element at position \e i.
       *
       * \param i Index position.
       * \param x New value of an element.
       */
      void setElement( const Index& i, const Value& x );

      /**
       * \brief Accesses specified element at the position \e i and returns its value.
       *
       * \param i Index position of an element.
       */
      Value getElement( const Index& i ) const;

      /**
       * \brief Accesses specified element at the position \e i and returns a reference to its value.
       *
       * \param i Index position of an element.
       */
      __cuda_callable__ inline Value& operator[] ( const Index& i );

      /**
       * \brief Accesses specified element at the position \e i and returns a (constant?) reference to its value.
       *
       * \param i Index position of an element.
       */
      __cuda_callable__ inline const Value& operator[] ( const Index& i ) const;

      /**
       * \brief Assigns \e array to this array, replacing its current contents.
       *
       * \param array Reference to an array.
       */
      Array& operator = ( const Array& array );

      /**
       * \brief Assigns \e array to this array, replacing its current contents.
       *
       * \param array Reference to an array.
       */
      Array& operator = ( Array&& array );

      /**
       * \brief Assigns \e array to this array, replacing its current contents.
       *
       * \tparam ArrayT Type of array.
       * \param array Reference to an array.
       */
      template< typename ArrayT >
      Array& operator = ( const ArrayT& array );

      /**
       * 
       * @param list
       * @return 
       */
      template< typename InValue >
      Array& operator = ( const std::list< InValue >& list );

      /**
       * 
       * @param vector
       * @return 
       */
      template< typename InValue >
      Array& operator = ( const std::vector< InValue >& vector );

      /**
       * \brief This function checks whether this array is equal to \e array.
       *
       * \tparam ArrayT Type of array.
       * \param array Reference to an array.
       */
      template< typename ArrayT >
      bool operator == ( const ArrayT& array ) const;

      /**
       * \brief This function checks whether this array is not equal to \e array.
       *
       * \tparam ArrayT Type of array.
       * \param array Reference to an array.
       */
      template< typename ArrayT >
      bool operator != ( const ArrayT& array ) const;

      /**
       * \brief Sets the array elements to given value.
       *
       * Sets all the array values to \e v.
       *
       * \param v Reference to a value.
       */
      void setValue( const Value& v,
                     const Index begin = 0,
                     Index end = -1 );

      /**
       * \brief Sets the array elements using given lambda function.
       *
       * Sets all the array values to \e v.
       *
       * \param v Reference to a value.
       */
      template< typename Function >
      void evaluate( Function& f,
                     const Index begin = 0,
                     Index end = -1 );

      /**
       * \brief Checks if there is an element with value \e v in this array.
       *
       * \param v Reference to a value.
       */
      bool containsValue( const Value& v,
                          const Index begin = 0,
                          Index end = -1 ) const;

      /**
       * \brief Checks if all elements in this array have the same value \e v.
       *
       * \param v Reference to a value.
       */
      bool containsOnlyValue( const Value& v,
                              const Index begin = 0,
                              Index end = -1 ) const;

      /**
       * \brief Returns true if non-zero size is set.
       */
      operator bool() const;

      /**
       * \brief Method for saving the object to a \e file as a binary data.
       *
       * \param file Reference to a file.
       */
      void save( File& file ) const;

      /**
       * Method for loading the object from a file as a binary data.
       *
       * \param file Reference to a file.
       */
      void load( File& file );

      /**
       * \brief This method loads data without reallocation.
       *
       * This is useful for loading data into shared arrays.
       * If the array was not initialize yet, common load is
       * performed. Otherwise, the array size must fit with
       * the size of array being loaded.
       */
      void boundLoad( File& file );

      using Object::save;

      using Object::load;

      using Object::boundLoad;

      /** \brief Basic destructor. */
      ~Array();

   protected:

      /** \brief Method for releasing array data. */
      void releaseData() const;

      /** \brief Number of elements in array. */
      mutable Index size = 0;

      /** \brief Pointer to data. */
      mutable Value* data = nullptr;

      /**
       * \brief Pointer to the originally allocated data.
       *
       * They might differ if one long array is partitioned into more shorter
       * arrays. Each of them must know the pointer on allocated data because
       * the last one must deallocate the array. If outer data (not allocated
       * by TNL) are bind then this pointer is zero since no deallocation is
       * necessary.
       */
      mutable Value* allocationPointer = nullptr;

      /**
       * \brief Counter of objects sharing this array or some parts of it.
       *
       * The reference counter is allocated after first sharing of the data between
       * more arrays. This is to avoid unnecessary dynamic memory allocation.
       */
      mutable int* referenceCounter = nullptr;
};

template< typename Array,
          typename Data,
          bool isArray = TNL::isArray< Data >::value >
struct ArrayAssignment {};

template< typename Array,
          typename Data >
struct ArrayAssignment< Array, Data, true >
{
   static void assign( Array& a, const Data& d );
};

template< typename Array,
          typename Data >
struct ArrayAssignment< Array, Data, false >
{
   static void assign( Array& a, const Data& d );
};

template< typename Value, typename Device, typename Index >
std::ostream& operator << ( std::ostream& str, const Array< Value, Device, Index >& v );

} // namespace Containers

template< typename Value,
          typename Device,
          typename Index >
struct isArray< Containers::Array< Value, Device, Index > >
{
   static constexpr bool value = true;
};

} // namespace TNL

#include <TNL/Containers/Array.hpp>
